<?php
require_once 'config/config.php';

$database = new Database();
$db = $database->getConnection();

// Pagination setup
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$jobs_per_page = 2;
$offset = ($page - 1) * $jobs_per_page;

// Count total active jobs
$stmt = $db->prepare("
    SELECT COUNT(*) as total 
    FROM jobs j 
    LEFT JOIN companies c ON j.company_id = c.id 
    WHERE j.status = 'active' AND c.status = 'approved'
");
$stmt->execute();
$total_jobs = $stmt->fetch()['total'];
$total_pages = ceil($total_jobs / $jobs_per_page);

// Get recent jobs with pagination
$stmt = $db->prepare("
    SELECT j.*, c.name as company_name, c.logo as company_logo, cat.name as category_name 
    FROM jobs j 
    LEFT JOIN companies c ON j.company_id = c.id 
    LEFT JOIN job_categories cat ON j.category_id = cat.id 
    WHERE j.status = 'active' AND c.status = 'approved'
    ORDER BY j.created_at DESC 
    LIMIT $offset, $jobs_per_page
");
$stmt->execute();
$recent_jobs = $stmt->fetchAll();

// Get job categories
$stmt = $db->prepare("SELECT * FROM job_categories ORDER BY name");
$stmt->execute();
$categories = $stmt->fetchAll();

// Get counts for animated counters
$stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs WHERE status = 'active'");
$stmt->execute();
$jobs_count = $stmt->fetch()['count'];

$stmt = $db->prepare("SELECT COUNT(*) as count FROM candidates");
$stmt->execute();
$candidates_count = $stmt->fetch()['count'];

$stmt = $db->prepare("SELECT COUNT(*) as count FROM companies WHERE status = 'approved'");
$stmt->execute();
$companies_count = $stmt->fetch()['count'];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo SITE_NAME; ?> - Find Your Dream Job</title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/x-icon" href="assets/images/favicon.ico">
    <link rel="apple-touch-icon" sizes="180x180" href="assets/images/favicon.png">
    <link rel="icon" type="image/png" sizes="32x32" href="assets/images/favicon-32x32.png">
    <link rel="icon" type="image/png" sizes="16x16" href="assets/images/favicon-16x16.png">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
    <style>
        .job-card-modern {
            transition: transform 0.2s ease-in-out;
        }
        
        .job-card-modern:hover {
            transform: translateY(-2px);
        }
        
        .job-item {
            border-radius: 8px !important;
            border: 1px solid #e3e6f0 !important;
            transition: all 0.3s ease;
        }
        
        .job-item:hover {
            border-color: #4285f4 !important;
            box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15) !important;
        }
        
        .job-title {
            font-size: 1.25rem;
            line-height: 1.4;
            margin-bottom: 0.5rem;
        }
        
        .job-title a {
            color: #4285f4 !important;
            text-decoration: none;
        }
        
        .job-title a:hover {
            color: #1a73e8 !important;
            text-decoration: underline;
        }
        
        .company-logo-small {
            width: 20px;
            height: 20px;
            object-fit: contain;
            border-radius: 2px;
        }
        
        .company-logo-small-placeholder {
            width: 20px;
            height: 20px;
            background-color: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 2px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 10px;
            color: #6c757d;
        }
        
        .job-description-preview {
            font-size: 0.9rem;
            line-height: 1.5;
        }
        
        .salary-info {
            font-size: 0.9rem;
        }
        
        .experience-info {
            font-size: 0.85rem;
        }
        
        .save-job-btn {
            border-radius: 50%;
            width: 36px;
            height: 36px;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s ease;
        }
        
        .save-job-btn:hover {
            background-color: #dc3545;
            border-color: #dc3545;
            color: white;
        }
        
        .save-job-btn:hover i {
            transform: scale(1.1);
        }
        
        .save-job-btn.saved {
            background-color: #dc3545;
            border-color: #dc3545;
            color: white;
        }
        
        .save-job-btn.saved i::before {
            content: "\f004"; /* solid heart */
        }
        
        /* Share Job Button */
        .share-job-btn {
            border-radius: 50%;
            width: 36px;
            height: 36px;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s ease;
        }
        
        .share-job-btn:hover {
            background-color: #28a745;
            border-color: #28a745;
            color: white;
        }
        
        .share-job-btn:hover i {
            transform: scale(1.1);
        }
        
        .share-dropdown {
            min-width: 160px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            border: none;
            border-radius: 8px;
            padding: 8px 0;
        }
        
        .share-dropdown .dropdown-item {
            padding: 8px 16px;
            font-size: 0.9rem;
            transition: all 0.2s ease;
        }
        
        .share-dropdown .dropdown-item:hover {
            background-color: #f8f9fa;
            transform: translateX(2px);
        }
        
        .share-dropdown .dropdown-item i {
            width: 18px;
        }
        
        .jobs-container-home {
            max-width: 100%;
        }
        
        @media (min-width: 768px) {
            .jobs-container-home {
                max-width: 900px;
                margin: 0 auto;
            }
        }
        
        .company-logo {
            width: 50px;
            height: 50px;
            object-fit: contain;
            border-radius: 8px;
            border: 1px solid #e9ecef;
            background: white;
            padding: 4px;
        }
        
        .company-logo-placeholder {
            width: 50px;
            height: 50px;
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #6c757d;
        }
        
        /* Mini Counters */
        .counter-mini-box {
            padding: 15px 10px;
            border-radius: 8px;
            background: #100d6e;
            color: white;
            box-shadow: 0 2px 8px rgba(16, 13, 110, 0.3);
            transition: transform 0.2s ease;
            opacity: 0;
            transform: translateY(20px);
            animation: fadeInUp 0.6s ease forwards;
        }
        
        .counter-mini-box:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(16, 13, 110, 0.4);
        }
        
        .counter-mini-box:nth-child(1) { animation-delay: 0.1s; }
        .counter-mini-box:nth-child(2) { animation-delay: 0.2s; }
        .counter-mini-box:nth-child(3) { animation-delay: 0.3s; }
        
        .counter-number-mini {
            font-size: 1.8rem;
            font-weight: 700;
            line-height: 1;
            display: block;
            margin-bottom: 4px;
        }
        
        .counter-label-mini {
            font-size: 0.75rem;
            font-weight: 500;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            opacity: 0.9;
        }
        
        @keyframes fadeInUp {
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="index.php">
                
                <img src="assets\images\cefgroup.png" alt="sanpc logo" width="350" height="100">
            </a>
            
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav me-auto">
                    <li class="nav-item">
                        <a class="nav-link active" href="index.php">Home</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="jobs.php">Browse Jobs</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="companies.php">Companies</a>
                    </li>
                </ul>
                
                <ul class="navbar-nav">
                    <?php if (isLoggedIn()): ?>
                        <?php if ($_SESSION['user_type'] === 'admin'): ?>
                            <li class="nav-item">
                                <a class="nav-link" href="admin/dashboard.php">Admin Dashboard</a>
                            </li>
                        <?php elseif ($_SESSION['user_type'] === 'company'): ?>
                            <li class="nav-item">
                                <a class="nav-link" href="company/dashboard.php">Company Dashboard</a>
                            </li>
                        <?php elseif ($_SESSION['user_type'] === 'candidate'): ?>
                            <li class="nav-item">
                                <a class="nav-link" href="candidate/dashboard.php">My Profile</a>
                            </li>
                        <?php endif; ?>
                        <li class="nav-item">
                            <a class="nav-link" href="auth/logout.php">Logout</a>
                        </li>
                    <?php else: ?>
                        <li class="nav-item">
                            <a class="nav-link" href="auth/login.php">Login</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="auth/register.php">Register</a>
                        </li>
                    <?php endif; ?>
                </ul>
            </div>
        </div>
    </nav>

    <!-- Hero Section -->
    <section class="hero-section text-white">
        <div class="container">
            <div class="row align-items-center min-vh-75">
                <div class="col-lg-8 mx-auto text-center hero-content">
                    <h1 class="display-3 fw-bold mb-4">Find Your Dream Job Today</h1>
                    <p class="lead mb-5 fs-4">Connect with top companies and discover amazing career opportunities that match your skills and aspirations in a modern, professional environment.</p>
                    <div class="d-flex flex-column flex-sm-row gap-3 justify-content-center">
                        <a href="jobs.php" class="btn btn-light btn-lg px-5 py-3">
                            <i class="fas fa-search me-2"></i>Browse Jobs
                        </a>
                        <a href="auth/register.php" class="btn btn-outline-light btn-lg px-5 py-3">
                            <i class="fas fa-user-plus me-2"></i>Join as Candidate
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Job Search Section -->
    <section class="py-5">
        <div class="container">
            <div class="row">
                <div class="col-lg-8">
                    <div class="card shadow-sm">
                        <div class="card-body p-4">
                            <form action="jobs.php" method="GET">
                                <div class="row g-3">
                                    <div class="col-md-4">
                                        <input type="text" class="form-control" name="keyword" placeholder="Job title, skills...">
                                    </div>
                                    <div class="col-md-3">
                                        <select class="form-select" name="category">
                                            <option value="">All Categories</option>
                                            <?php foreach ($categories as $category): ?>
                                                <option value="<?php echo $category['id']; ?>"><?php echo htmlspecialchars($category['name']); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div class="col-md-3">
                                        <input type="text" class="form-control" name="location" placeholder="Location">
                                    </div>
                                    <div class="col-md-2">
                                        <button type="submit" class="btn btn-primary w-100">
                                            <i class="fas fa-search"></i> Search
                                        </button>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
                
                <!-- Counters on the right -->
                <div class="col-lg-4">
                    <div class="row h-100 align-items-center">
                        <div class="col-12">
                            <div class="row text-center">
                                <div class="col-4">
                                    <div class="counter-mini-box">
                                        <div class="counter-number-mini" data-count="<?php echo $jobs_count; ?>">0</div>
                                        <div class="counter-label-mini">Jobs</div>
                                    </div>
                                </div>
                                <div class="col-4">
                                    <div class="counter-mini-box">
                                        <div class="counter-number-mini" data-count="<?php echo $candidates_count; ?>">0</div>
                                        <div class="counter-label-mini">Candidates</div>
                                    </div>
                                </div>
                                <div class="col-4">
                                    <div class="counter-mini-box">
                                        <div class="counter-number-mini" data-count="<?php echo $companies_count; ?>">0</div>
                                        <div class="counter-label-mini">Companies</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Recent Jobs Section -->
    <section class="py-5 bg-light">
        <div class="container">
            <div class="text-center mb-5">
                <h2 class="fw-bold">Job Opportunities</h2>
                <p class="text-muted">Browse through <?php echo $total_jobs; ?> available positions from top companies</p>
                <?php if ($total_pages > 1): ?>
                    <p class="text-sm text-muted">Page <?php echo $page; ?> of <?php echo $total_pages; ?></p>
                <?php endif; ?>
            </div>
            
            <div class="jobs-container-home">
                <?php foreach ($recent_jobs as $job): ?>
                    <div class="job-card-modern mb-3">
                        <div class="card border-0 shadow-sm job-item">
                            <div class="card-body p-4">
                                <div class="d-flex justify-content-between align-items-start">
                                    <div class="flex-grow-1">
                                        <!-- Job Title -->
                                        <h3 class="job-title mb-2">
                                            <a href="job-details.php?id=<?php echo $job['id']; ?>" class="text-decoration-none">
                                                <?php echo htmlspecialchars($job['title']); ?>
                                            </a>
                                        </h3>
                                        
                                        <!-- Company and Location -->
                                        <div class="d-flex align-items-center text-muted mb-3">
                                            <div class="company-info d-flex align-items-center me-4">
                                                <?php if ($job['company_logo']): ?>
                                                    <img src="<?php echo UPLOAD_URL . 'logos/' . $job['company_logo']; ?>" 
                                                         alt="<?php echo htmlspecialchars($job['company_name']); ?>" 
                                                         class="company-logo-small me-2">
                                                <?php else: ?>
                                                    <div class="company-logo-small-placeholder me-2">
                                                        <i class="fas fa-building"></i>
                                                    </div>
                                                <?php endif; ?>
                                                <span><?php echo htmlspecialchars($job['company_name']); ?></span>
                                            </div>
                                            <div class="location-info d-flex align-items-center">
                                                <i class="fas fa-map-marker-alt me-1"></i>
                                                <span><?php echo htmlspecialchars($job['location']); ?></span>
                                            </div>
                                        </div>
                                        
                                        <!-- Job Description Preview -->
                                        <p class="job-description-preview text-muted mb-3">
                                            <?php 
                                                $description = strip_tags($job['description']);
                                                echo htmlspecialchars(strlen($description) > 150 ? substr($description, 0, 150) . '...' : $description);
                                            ?>
                                        </p>
                                        
                                        <!-- Salary and Experience -->
                                        <div class="d-flex flex-wrap gap-3 mb-3">
                                            <?php if ($job['salary_min'] || $job['salary_max']): ?>
                                                <div class="salary-info">
                                                    <i class="fas fa-money-bill-wave text-success me-1"></i>
                                                    <span class="fw-semibold text-success">
                                                        <?php if ($job['salary_min'] && $job['salary_max']): ?>
                                                            R<?php echo number_format($job['salary_min']); ?> - R<?php echo number_format($job['salary_max']); ?>
                                                        <?php elseif ($job['salary_min']): ?>
                                                            From R<?php echo number_format($job['salary_min']); ?>
                                                        <?php else: ?>
                                                            Up to R<?php echo number_format($job['salary_max']); ?>
                                                        <?php endif; ?>
                                                    </span>
                                                </div>
                                            <?php endif; ?>
                                            <div class="experience-info text-muted">
                                                <?php echo ucfirst($job['experience_level']); ?>+ years experience in <?php echo htmlspecialchars($job['category_name'] ?: 'General'); ?>
                                            </div>
                                        </div>
                                        
                                        <!-- Posted Time -->
                                        <div class="text-muted small">
                                            <?php 
                                                $posted_time = new DateTime($job['created_at']);
                                                $now = new DateTime();
                                                $interval = $now->diff($posted_time);
                                                
                                                if ($interval->d > 0) {
                                                    echo $interval->d . ' day' . ($interval->d > 1 ? 's' : '') . ' ago';
                                                } elseif ($interval->h > 0) {
                                                    echo $interval->h . ' hour' . ($interval->h > 1 ? 's' : '') . ' ago';
                                                } else {
                                                    echo 'Just posted';
                                                }
                                            ?>
                                        </div>
                                    </div>
                                    
                                    <!-- Save Job & Share Buttons -->
                                    <div class="job-actions d-flex gap-2">
                                        <button class="btn btn-outline-secondary btn-sm save-job-btn" data-job-id="<?php echo $job['id']; ?>" title="Save Job">
                                            <i class="far fa-heart"></i>
                                        </button>
                                        
                                        <!-- Share Button with Dropdown -->
                                        <div class="dropdown">
                                            <button class="btn btn-outline-secondary btn-sm share-job-btn" data-bs-toggle="dropdown" aria-expanded="false" title="Share Job">
                                                <i class="fas fa-share-alt"></i>
                                            </button>
                                            <ul class="dropdown-menu dropdown-menu-end share-dropdown">
                                                <li>
                                                    <a class="dropdown-item share-option" 
                                                       href="#" 
                                                       data-platform="whatsapp" 
                                                       data-url="<?php echo 'http://' . $_SERVER['HTTP_HOST'] . '/recruitment/job-details.php?id=' . $job['id']; ?>"
                                                       data-title="<?php echo htmlspecialchars($job['title']); ?>"
                                                       data-company="<?php echo htmlspecialchars($job['company_name']); ?>">
                                                        <i class="fab fa-whatsapp text-success me-2"></i>WhatsApp
                                                    </a>
                                                </li>
                                                <li>
                                                    <a class="dropdown-item share-option" 
                                                       href="#" 
                                                       data-platform="facebook" 
                                                       data-url="<?php echo 'http://' . $_SERVER['HTTP_HOST'] . '/recruitment/job-details.php?id=' . $job['id']; ?>"
                                                       data-title="<?php echo htmlspecialchars($job['title']); ?>"
                                                       data-company="<?php echo htmlspecialchars($job['company_name']); ?>">
                                                        <i class="fab fa-facebook-f text-primary me-2"></i>Facebook
                                                    </a>
                                                </li>
                                                <li>
                                                    <a class="dropdown-item share-option" 
                                                       href="#" 
                                                       data-platform="twitter" 
                                                       data-url="<?php echo 'http://' . $_SERVER['HTTP_HOST'] . '/recruitment/job-details.php?id=' . $job['id']; ?>"
                                                       data-title="<?php echo htmlspecialchars($job['title']); ?>"
                                                       data-company="<?php echo htmlspecialchars($job['company_name']); ?>">
                                                        <i class="fab fa-twitter text-info me-2"></i>X (Twitter)
                                                    </a>
                                                </li>
                                                <li>
                                                    <a class="dropdown-item share-option" 
                                                       href="#" 
                                                       data-platform="email" 
                                                       data-url="<?php echo 'http://' . $_SERVER['HTTP_HOST'] . '/recruitment/job-details.php?id=' . $job['id']; ?>"
                                                       data-title="<?php echo htmlspecialchars($job['title']); ?>"
                                                       data-company="<?php echo htmlspecialchars($job['company_name']); ?>">
                                                        <i class="fas fa-envelope text-secondary me-2"></i>Email
                                                    </a>
                                                </li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            
            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
                <nav aria-label="Job listings pagination" class="mt-4">
                    <ul class="pagination justify-content-center">
                        <?php if ($page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=<?php echo $page - 1; ?>">
                                    <i class="fas fa-chevron-left me-1"></i>Previous
                                </a>
                            </li>
                        <?php endif; ?>
                        
                        <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                            <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                                <a class="page-link" href="?page=<?php echo $i; ?>"><?php echo $i; ?></a>
                            </li>
                        <?php endfor; ?>
                        
                        <?php if ($page < $total_pages): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=<?php echo $page + 1; ?>">
                                    Next<i class="fas fa-chevron-right ms-1"></i>
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </nav>
            <?php endif; ?>
            
            <div class="text-center mt-3">
                <p class="text-muted">Showing <?php echo count($recent_jobs); ?> of <?php echo $total_jobs; ?> available jobs</p>
                <a href="jobs.php" class="btn btn-outline-primary">
                    <i class="fas fa-search me-1"></i>Browse All Jobs with Filters
                </a>
            </div>
        </div>
    </section>

    <!-- Features Section -->
    <section class="py-5">
        <div class="container">
            <div class="text-center mb-5">
                <h2 class="fw-bold">Why Choose Our Platform?</h2>
            </div>
            
            <div class="row">
                <div class="col-lg-4 col-md-6 mb-4">
                    <div class="text-center">
                        <div class="feature-icon bg-primary text-white rounded-circle mx-auto mb-3">
                            <i class="fas fa-search fa-2x"></i>
                        </div>
                        <h4>Easy Job Search</h4>
                        <p class="text-muted">Find the perfect job with our advanced search and filtering options.</p>
                    </div>
                </div>
                
                <div class="col-lg-4 col-md-6 mb-4">
                    <div class="text-center">
                        <div class="feature-icon bg-success text-white rounded-circle mx-auto mb-3">
                            <i class="fas fa-users fa-2x"></i>
                        </div>
                        <h4>Top Companies</h4>
                        <p class="text-muted">Connect with leading companies and employers in your industry.</p>
                    </div>
                </div>
                
                <div class="col-lg-4 col-md-6 mb-4">
                    <div class="text-center">
                        <div class="feature-icon bg-info text-white rounded-circle mx-auto mb-3">
                            <i class="fas fa-file-alt fa-2x"></i>
                        </div>
                        <h4>Online CV Builder</h4>
                        <p class="text-muted">Create professional resumes with our built-in CV builder tool.</p>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Footer -->
    <footer class="bg-dark text-white py-4">
        <div class="container">
            <div class="row">
                <div class="col-lg-6">
                    <h5><?php echo SITE_NAME; ?></h5>
                    <p class="text-muted">Connecting talented individuals with amazing opportunities.</p>
                </div>
                <div class="col-lg-6">
                    <div class="text-lg-end">
                        <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
                    </div>
                </div>
            </div>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Handle hero background image fallbacks for office team image
        document.addEventListener('DOMContentLoaded', function() {
            const heroSection = document.querySelector('.hero-section');
            if (heroSection) {
                // Test if hero-office-team.jpg loads successfully
                const testImage = new Image();
                testImage.onload = function() {
                    console.log('Office team hero image loaded successfully as background');
                };
                testImage.onerror = function() {
                    console.warn('Office team hero image failed to load, using fallback');
                    // Fallback to orange gradient if local image fails
                    heroSection.style.background = `
                        linear-gradient(135deg, #FFA500 0%, #FF8C00 100%)
                    `;
                };
                testImage.src = 'assets/images/hero-office-team.jpg';
            }
        });
        
        // Mini Counters Animation
        function animateMiniCounters() {
            const counters = document.querySelectorAll('.counter-number-mini');
            
            counters.forEach((counter, index) => {
                const target = parseInt(counter.getAttribute('data-count'));
                const duration = 1500; // 1.5 seconds
                const start = Date.now();
                const startValue = 0;
                
                function updateCounter() {
                    const now = Date.now();
                    const elapsed = now - start;
                    const progress = Math.min(elapsed / duration, 1);
                    
                    // Use easing function for smooth animation
                    const easeOutExpo = progress === 1 ? 1 : 1 - Math.pow(2, -10 * progress);
                    const currentValue = Math.floor(startValue + (target - startValue) * easeOutExpo);
                    
                    counter.textContent = currentValue.toLocaleString();
                    
                    if (progress < 1) {
                        requestAnimationFrame(updateCounter);
                    } else {
                        counter.textContent = target.toLocaleString();
                    }
                }
                
                // Start animation with staggered delay
                setTimeout(() => {
                    updateCounter();
                }, 500 + (index * 100));
            });
        }
        
        // Intersection Observer to trigger animation when counters come into view
        function initMiniCounterAnimation() {
            const counterSection = document.querySelector('.col-lg-4');
            
            if (counterSection) {
                const observer = new IntersectionObserver((entries) => {
                    entries.forEach(entry => {
                        if (entry.isIntersecting) {
                            animateMiniCounters();
                            observer.unobserve(entry.target);
                        }
                    });
                }, {
                    threshold: 0.3
                });
                
                observer.observe(counterSection);
            }
        }
        
        // Save job functionality
        document.addEventListener('DOMContentLoaded', function() {
            // Initialize mini counter animation
            initMiniCounterAnimation();
            
            document.querySelectorAll('.save-job-btn').forEach(button => {
                button.addEventListener('click', function(e) {
                    e.preventDefault();
                    const jobId = this.dataset.jobId;
                    const icon = this.querySelector('i');
                    
                    // Toggle saved state
                    if (this.classList.contains('saved')) {
                        this.classList.remove('saved');
                        icon.className = 'far fa-heart';
                        this.title = 'Save Job';
                    } else {
                        this.classList.add('saved');
                        icon.className = 'fas fa-heart';
                        this.title = 'Job Saved';
                    }
                    
                    // Here you could add AJAX call to save/unsave job
                    // fetch('/api/save-job.php', { method: 'POST', ... })
                });
            });
            
            // Share job functionality
            document.querySelectorAll('.share-option').forEach(shareLink => {
                shareLink.addEventListener('click', function(e) {
                    e.preventDefault();
                    
                    const platform = this.getAttribute('data-platform');
                    const url = this.getAttribute('data-url');
                    const title = this.getAttribute('data-title');
                    const company = this.getAttribute('data-company');
                    
                    const shareText = `Check out this ${title} position at ${company}!`;
                    
                    let shareUrl = '';
                    
                    switch(platform) {
                        case 'whatsapp':
                            shareUrl = `https://wa.me/?text=${encodeURIComponent(shareText + ' ' + url)}`;
                            break;
                        case 'facebook':
                            shareUrl = `https://www.facebook.com/sharer/sharer.php?u=${encodeURIComponent(url)}&quote=${encodeURIComponent(shareText)}`;
                            break;
                        case 'twitter':
                            shareUrl = `https://twitter.com/intent/tweet?text=${encodeURIComponent(shareText)}&url=${encodeURIComponent(url)}`;
                            break;
                        case 'email':
                            shareUrl = `mailto:?subject=${encodeURIComponent('Job Opportunity: ' + title)}&body=${encodeURIComponent(shareText + '\n\n' + url)}`;
                            break;
                    }
                    
                    if (shareUrl) {
                        if (platform === 'email') {
                            window.location.href = shareUrl;
                        } else {
                            window.open(shareUrl, '_blank', 'width=600,height=400,scrollbars=yes,resizable=yes');
                        }
                    }
                });
            });
        });
    </script>
</body>
</html>