<?php
require_once '../config/config.php';
requireLogin('candidate');

$database = new Database();
$db = $database->getConnection();

$candidate_id = $_SESSION['user_id'];

$error = '';
$success = '';

// Get candidate details
$stmt = $db->prepare("SELECT * FROM candidates WHERE id = ?");
$stmt->execute([$candidate_id]);
$candidate = $stmt->fetch();

if (!$candidate) {
    redirect('../auth/logout.php');
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $first_name = sanitize($_POST['first_name']);
    $last_name = sanitize($_POST['last_name']);
    $email = sanitize($_POST['email']);
    $phone = sanitize($_POST['phone']);
    $address = sanitize($_POST['address']);
    $date_of_birth = $_POST['date_of_birth'];
    $gender = $_POST['gender'];
    $nationality = sanitize($_POST['nationality']);
    $id_number = null;
    $passport_number = null;
    $password = $_POST['password'];
    
    // Handle nationality and identification
    if ($nationality === 'south_african') {
        $id_number = sanitize($_POST['id_number']);
        if (empty($id_number)) {
            $error = 'South African ID number is required.';
        } elseif (!preg_match('/^[0-9]{13}$/', $id_number)) {
            $error = 'South African ID number must be exactly 13 digits.';
        }
    } elseif ($nationality === 'foreign') {
        $passport_number = sanitize($_POST['passport_number']);
        if (empty($passport_number)) {
            $error = 'Passport number is required for foreign nationals.';
        }
    }
    
    if (empty($first_name) || empty($last_name) || empty($email)) {
        $error = 'First name, last name, and email are required.';
    } else {
        // Check if email is already used by someone else
        $check_tables = [
            ['table' => 'candidates', 'email_field' => 'email'],
            ['table' => 'company_users', 'email_field' => 'email'],
            ['table' => 'admins', 'email_field' => 'email']
        ];
        
        $email_exists = false;
        foreach ($check_tables as $check) {
            $stmt = $db->prepare("SELECT id FROM {$check['table']} WHERE {$check['email_field']} = ? AND id != ?");
            $params = [$email];
            if ($check['table'] === 'candidates') {
                $params[] = $candidate_id;
            } else {
                $params[] = 0; // For other tables, just check if email exists
            }
            $stmt->execute($params);
            if ($stmt->fetch()) {
                $email_exists = true;
                break;
            }
        }
        
        if ($email_exists) {
            $error = 'Email address is already in use.';
        } else {
            try {
                // Handle profile picture upload
                $profile_picture = $candidate['profile_picture'];
                if (isset($_FILES['profile_picture']) && $_FILES['profile_picture']['error'] === UPLOAD_ERR_OK) {
                    $upload_dir = '../uploads/profiles/';
                    
                    if (!file_exists($upload_dir)) {
                        mkdir($upload_dir, 0777, true);
                    }
                    
                    $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
                    $file_type = $_FILES['profile_picture']['type'];
                    
                    if (in_array($file_type, $allowed_types)) {
                        $file_extension = pathinfo($_FILES['profile_picture']['name'], PATHINFO_EXTENSION);
                        $new_filename = uniqid() . '.' . $file_extension;
                        $upload_path = $upload_dir . $new_filename;
                        
                        if (move_uploaded_file($_FILES['profile_picture']['tmp_name'], $upload_path)) {
                            // Delete old profile picture if it exists
                            if ($profile_picture && file_exists($upload_dir . $profile_picture)) {
                                unlink($upload_dir . $profile_picture);
                            }
                            $profile_picture = $new_filename;
                        } else {
                            $error = 'Failed to upload profile picture.';
                        }
                    } else {
                        $error = 'Profile picture must be a JPEG, PNG, or GIF image.';
                    }
                }
                
                if (!$error) {
                    if (!empty($password)) {
                        // Update with new password
                        if (strlen($password) < 6) {
                            $error = 'Password must be at least 6 characters long.';
                        } else {
                            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                            $stmt = $db->prepare("
                                UPDATE candidates 
                                SET first_name = ?, last_name = ?, email = ?, phone = ?, address = ?, 
                                    date_of_birth = ?, gender = ?, nationality = ?, id_number = ?, passport_number = ?, 
                                    profile_picture = ?, password = ?
                                WHERE id = ?
                            ");
                            $stmt->execute([
                                $first_name, $last_name, $email, $phone, $address, 
                                $date_of_birth ?: null, $gender ?: null, $nationality, $id_number, $passport_number,
                                $profile_picture, $hashed_password, $candidate_id
                            ]);
                        }
                    } else {
                        // Update without changing password
                        $stmt = $db->prepare("
                            UPDATE candidates 
                            SET first_name = ?, last_name = ?, email = ?, phone = ?, address = ?, 
                                date_of_birth = ?, gender = ?, nationality = ?, id_number = ?, passport_number = ?, 
                                profile_picture = ?
                            WHERE id = ?
                        ");
                        $stmt->execute([
                            $first_name, $last_name, $email, $phone, $address, 
                            $date_of_birth ?: null, $gender ?: null, $nationality, $id_number, $passport_number,
                            $profile_picture, $candidate_id
                        ]);
                    }
                    
                    if (!$error) {
                        $success = 'Profile updated successfully!';
                        
                        // Update session variables
                        $_SESSION['user_email'] = $email;
                        $_SESSION['first_name'] = $first_name;
                        $_SESSION['last_name'] = $last_name;
                        
                        // Refresh candidate data
                        $stmt = $db->prepare("SELECT * FROM candidates WHERE id = ?");
                        $stmt->execute([$candidate_id]);
                        $candidate = $stmt->fetch();
                    }
                }
            } catch (PDOException $e) {
                $error = 'Error updating profile. Please try again.';
            }
        }
    }
}

// Check profile completion including CV builder sections
$profile_completion = 0;
$total_sections = 14; // Basic profile (9) + CV sections (5)
$completed_sections = 0;

// Basic profile fields
if (!empty($candidate['first_name'])) $completed_sections++;
if (!empty($candidate['last_name'])) $completed_sections++;
if (!empty($candidate['email'])) $completed_sections++;
if (!empty($candidate['phone'])) $completed_sections++;
if (!empty($candidate['address'])) $completed_sections++;
if (!empty($candidate['date_of_birth'])) $completed_sections++;
if (!empty($candidate['gender'])) $completed_sections++;
if (!empty($candidate['profile_picture'])) $completed_sections++;
// Nationality and identification
if (!empty($candidate['nationality']) && 
    (($candidate['nationality'] === 'south_african' && !empty($candidate['id_number'])) || 
     ($candidate['nationality'] === 'foreign' && !empty($candidate['passport_number'])))) $completed_sections++;

// CV Builder sections - check if each section has at least one entry
// Professional Summary & Skills (resumes table)
$stmt = $db->prepare("SELECT COUNT(*) as count FROM resumes WHERE candidate_id = ? AND (summary IS NOT NULL AND summary != '') AND (skills IS NOT NULL AND skills != '')");
$stmt->execute([$candidate_id]);
$resume_count = $stmt->fetch()['count'];
if ($resume_count > 0) $completed_sections++;

// Education section
$stmt = $db->prepare("SELECT COUNT(*) as count FROM education WHERE candidate_id = ?");
$stmt->execute([$candidate_id]);
$education_count = $stmt->fetch()['count'];
if ($education_count > 0) $completed_sections++;

// Work Experience section
$stmt = $db->prepare("SELECT COUNT(*) as count FROM work_experience WHERE candidate_id = ?");
$stmt->execute([$candidate_id]);
$experience_count = $stmt->fetch()['count'];
if ($experience_count > 0) $completed_sections++;

// Certificates section
$stmt = $db->prepare("SELECT COUNT(*) as count FROM certificates WHERE candidate_id = ?");
$stmt->execute([$candidate_id]);
$certificates_count = $stmt->fetch()['count'];
if ($certificates_count > 0) $completed_sections++;

// Work References section
$stmt = $db->prepare("SELECT COUNT(*) as count FROM work_references WHERE candidate_id = ?");
$stmt->execute([$candidate_id]);
$references_count = $stmt->fetch()['count'];
if ($references_count > 0) $completed_sections++;

$profile_completion = round(($completed_sections / $total_sections) * 100);

// Get resumes count
$stmt = $db->prepare("SELECT COUNT(*) as resume_count FROM resumes WHERE candidate_id = ?");
$stmt->execute([$candidate_id]);
$resume_stats = $stmt->fetch();

// Get applications count
$stmt = $db->prepare("SELECT COUNT(*) as application_count FROM job_applications WHERE candidate_id = ?");
$stmt->execute([$candidate_id]);
$application_stats = $stmt->fetch();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Profile - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container-fluid">
            <a class="navbar-brand" href="../index.php">
                <i class="fas fa-briefcase me-2"></i><?php echo SITE_NAME; ?>
            </a>
            
            <div class="navbar-nav ms-auto">
                <span class="navbar-text me-3">
                    Welcome, <?php echo htmlspecialchars($_SESSION['first_name'] . ' ' . $_SESSION['last_name']); ?>
                </span>
                <a class="nav-link" href="../auth/logout.php">
                    <i class="fas fa-sign-out-alt me-1"></i>Logout
                </a>
            </div>
        </div>
    </nav>

    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <nav class="col-md-3 col-lg-2 sidebar">
                <div class="position-sticky">
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link" href="dashboard.php">
                                <i class="fas fa-tachometer-alt me-2"></i>Dashboard
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link active" href="profile.php">
                                <i class="fas fa-user me-2"></i>My Profile
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="cv-builder.php">
                                <i class="fas fa-file-alt me-2"></i>CV Builder
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="applications.php">
                                <i class="fas fa-paper-plane me-2"></i>My Applications
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="../jobs.php">
                                <i class="fas fa-search me-2"></i>Browse Jobs
                            </a>
                        </li>
                    </ul>
                </div>
            </nav>

            <!-- Main Content -->
            <main class="col-md-9 col-lg-10 px-md-4">
                <div class="py-4">
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h1 class="h2">
                            <i class="fas fa-user me-2"></i>My Profile
                        </h1>
                    </div>

                    <?php if ($error): ?>
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-circle me-2"></i><?php echo $error; ?>
                        </div>
                    <?php endif; ?>

                    <?php if ($success): ?>
                        <div class="alert alert-success">
                            <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
                        </div>
                    <?php endif; ?>

                    <div class="row">
                        <!-- Profile Form -->
                        <div class="col-lg-8">
                            <div class="card">
                                <div class="card-header">
                                    <h5 class="card-title mb-0">
                                        <i class="fas fa-edit me-2"></i>Personal Information
                                    </h5>
                                </div>
                                <div class="card-body">
                                    <form method="POST" enctype="multipart/form-data">
                                        <div class="row">
                                            <div class="col-md-8">
                                                <div class="row">
                                                    <div class="col-md-6 mb-3">
                                                        <label for="first_name" class="form-label">First Name *</label>
                                                        <input type="text" class="form-control" id="first_name" name="first_name" 
                                                               value="<?php echo htmlspecialchars($candidate['first_name']); ?>" required>
                                                    </div>
                                                    <div class="col-md-6 mb-3">
                                                        <label for="last_name" class="form-label">Last Name *</label>
                                                        <input type="text" class="form-control" id="last_name" name="last_name" 
                                                               value="<?php echo htmlspecialchars($candidate['last_name']); ?>" required>
                                                    </div>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <label for="email" class="form-label">Email Address *</label>
                                                    <input type="email" class="form-control" id="email" name="email" 
                                                           value="<?php echo htmlspecialchars($candidate['email']); ?>" required>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <label for="phone" class="form-label">Phone Number</label>
                                                    <input type="tel" class="form-control" id="phone" name="phone" 
                                                           value="<?php echo htmlspecialchars($candidate['phone']); ?>">
                                                </div>

                                                <div class="row">
                                                    <div class="col-md-6 mb-3">
                                                        <label for="date_of_birth" class="form-label">Date of Birth</label>
                                                        <input type="date" class="form-control" id="date_of_birth" name="date_of_birth" 
                                                               value="<?php echo $candidate['date_of_birth']; ?>" max="<?php echo date('Y-m-d', strtotime('-16 years')); ?>">
                                                    </div>
                                                    <div class="col-md-6 mb-3">
                                                        <label for="gender" class="form-label">Gender</label>
                                                        <select class="form-select" id="gender" name="gender">
                                                            <option value="">Select Gender</option>
                                                            <option value="male" <?php echo $candidate['gender'] === 'male' ? 'selected' : ''; ?>>Male</option>
                                                            <option value="female" <?php echo $candidate['gender'] === 'female' ? 'selected' : ''; ?>>Female</option>
                                                            <option value="other" <?php echo $candidate['gender'] === 'other' ? 'selected' : ''; ?>>Other</option>
                                                        </select>
                                                    </div>
                                                </div>

                                                <!-- Nationality & Identification Section -->
                                                <div class="card mb-3">
                                                    <div class="card-header bg-info text-white">
                                                        <h6 class="mb-0"><i class="fas fa-flag me-2"></i>Nationality & Identification</h6>
                                                    </div>
                                                    <div class="card-body">
                                                        <div class="mb-3">
                                                            <label class="form-label">Nationality *</label>
                                                            <div>
                                                                <div class="form-check form-check-inline">
                                                                    <input class="form-check-input" type="radio" name="nationality" id="south_african_edit" value="south_african" 
                                                                           <?php echo (!$candidate['nationality'] || $candidate['nationality'] === 'south_african') ? 'checked' : ''; ?>>
                                                                    <label class="form-check-label" for="south_african_edit">
                                                                        <i class="fas fa-flag me-1"></i>South African
                                                                    </label>
                                                                </div>
                                                                <div class="form-check form-check-inline">
                                                                    <input class="form-check-input" type="radio" name="nationality" id="foreign_edit" value="foreign"
                                                                           <?php echo ($candidate['nationality'] === 'foreign') ? 'checked' : ''; ?>>
                                                                    <label class="form-check-label" for="foreign_edit">
                                                                        <i class="fas fa-globe me-1"></i>Foreign National
                                                                    </label>
                                                                </div>
                                                            </div>
                                                        </div>
                                                        
                                                        <div id="south_african_id_edit" class="mb-3">
                                                            <label for="id_number_edit" class="form-label">South African ID Number *</label>
                                                            <input type="text" class="form-control" id="id_number_edit" name="id_number" 
                                                                   maxlength="13" pattern="[0-9]{13}"
                                                                   value="<?php echo htmlspecialchars($candidate['id_number']); ?>"
                                                                   placeholder="e.g. 9001015009087">
                                                            <div class="form-text">Enter your 13-digit South African ID number</div>
                                                        </div>
                                                        
                                                        <div id="foreign_passport_edit" class="mb-3" style="display: none;">
                                                            <label for="passport_number_edit" class="form-label">Passport Number *</label>
                                                            <input type="text" class="form-control" id="passport_number_edit" name="passport_number" 
                                                                   maxlength="20"
                                                                   value="<?php echo htmlspecialchars($candidate['passport_number']); ?>"
                                                                   placeholder="e.g. A12345678">
                                                            <div class="form-text">Enter your international passport number</div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="col-md-4 text-center">
                                                <div class="mb-3">
                                                    <label class="form-label">Profile Picture</label>
                                                    <div class="mb-3">
                                                        <?php if ($candidate['profile_picture']): ?>
                                                            <img src="<?php echo UPLOAD_URL . 'profiles/' . $candidate['profile_picture']; ?>" 
                                                                 alt="Profile Picture" class="profile-picture img-fluid">
                                                        <?php else: ?>
                                                            <div class="profile-picture bg-light d-flex align-items-center justify-content-center mx-auto">
                                                                <i class="fas fa-user text-muted fa-4x"></i>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                    <input type="file" class="form-control" name="profile_picture" accept="image/*">
                                                    <div class="form-text">Upload your photo (optional)</div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="address" class="form-label">Address</label>
                                            <textarea class="form-control" id="address" name="address" rows="3" 
                                                      placeholder="Your current address"><?php echo htmlspecialchars($candidate['address']); ?></textarea>
                                        </div>
                                        
                                        <div class="mb-4">
                                            <label for="password" class="form-label">New Password</label>
                                            <input type="password" class="form-control" id="password" name="password" 
                                                   placeholder="Leave blank to keep current password">
                                            <div class="form-text">Minimum 6 characters</div>
                                        </div>
                                        
                                        <button type="submit" class="btn btn-primary">
                                            <i class="fas fa-save me-1"></i>Update Profile
                                        </button>
                                    </form>
                                </div>
                            </div>
                        </div>

                        <!-- Profile Stats & Quick Actions -->
                        <div class="col-lg-4">
                            <!-- Profile Completion -->
                            <div class="card mb-4">
                                <div class="card-header">
                                    <h5 class="card-title mb-0">
                                        <i class="fas fa-chart-pie me-2"></i>Profile Completion
                                    </h5>
                                </div>
                                <div class="card-body text-center">
                                    <div class="mb-3">
                                        <div class="progress" style="height: 20px;">
                                            <div class="progress-bar" role="progressbar" 
                                                 style="width: <?php echo $profile_completion; ?>%" 
                                                 aria-valuenow="<?php echo $profile_completion; ?>" 
                                                 aria-valuemin="0" aria-valuemax="100">
                                                <?php echo $profile_completion; ?>%
                                            </div>
                                        </div>
                                    </div>
                                    <p class="text-muted">
                                        <?php echo $completed_sections; ?> of <?php echo $total_sections; ?> sections completed
                                    </p>
                                    <?php if ($profile_completion < 100): ?>
                                        <p class="text-warning small">
                                            <i class="fas fa-exclamation-triangle me-1"></i>
                                            Complete your profile and CV to increase visibility to employers
                                        </p>
                                        
                                        <!-- Missing sections details -->
                                        <div class="text-start small">
                                            <p class="mb-1"><strong>Missing sections:</strong></p>
                                            <?php if (empty($candidate['profile_picture'])): ?>
                                                <div class="text-muted">• Profile Picture</div>
                                            <?php endif; ?>
                                            <?php if (empty($candidate['address'])): ?>
                                                <div class="text-muted">• Address</div>
                                            <?php endif; ?>
                                            <?php if (empty($candidate['date_of_birth'])): ?>
                                                <div class="text-muted">• Date of Birth</div>
                                            <?php endif; ?>
                                            <?php if (empty($candidate['gender'])): ?>
                                                <div class="text-muted">• Gender</div>
                                            <?php endif; ?>
                                            <?php if (empty($candidate['nationality']) || 
                                                     (($candidate['nationality'] === 'south_african' && empty($candidate['id_number'])) || 
                                                      ($candidate['nationality'] === 'foreign' && empty($candidate['passport_number'])))): ?>
                                                <div class="text-muted">• Nationality & ID/Passport</div>
                                            <?php endif; ?>
                                            <?php if ($resume_count == 0): ?>
                                                <div class="text-muted">• Professional Summary & Skills</div>
                                            <?php endif; ?>
                                            <?php if ($education_count == 0): ?>
                                                <div class="text-muted">• Education History</div>
                                            <?php endif; ?>
                                            <?php if ($experience_count == 0): ?>
                                                <div class="text-muted">• Work Experience</div>
                                            <?php endif; ?>
                                            <?php if ($certificates_count == 0): ?>
                                                <div class="text-muted">• Certificates</div>
                                            <?php endif; ?>
                                            <?php if ($references_count == 0): ?>
                                                <div class="text-muted">• Work References</div>
                                            <?php endif; ?>
                                            <div class="mt-2">
                                                <a href="cv-builder.php" class="btn btn-sm btn-primary">
                                                    <i class="fas fa-edit me-1"></i>Complete CV Builder
                                                </a>
                                            </div>
                                        </div>
                                    <?php else: ?>
                                        <p class="text-success small">
                                            <i class="fas fa-check-circle me-1"></i>
                                            Your profile and CV are complete!
                                        </p>
                                    <?php endif; ?>
                                </div>
                            </div>

                            <!-- CV Builder Stats -->
                            <div class="card mb-4">
                                <div class="card-header">
                                    <h5 class="card-title mb-0">
                                        <i class="fas fa-chart-bar me-2"></i>CV Builder Status
                                    </h5>
                                </div>
                                <div class="card-body">
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <span><i class="fas fa-user-tie me-2 text-info"></i>Professional Summary</span>
                                        <span class="badge <?php echo $resume_count > 0 ? 'bg-success' : 'bg-secondary'; ?>">
                                            <?php echo $resume_count > 0 ? 'Complete' : 'Missing'; ?>
                                        </span>
                                    </div>
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <span><i class="fas fa-graduation-cap me-2 text-primary"></i>Education</span>
                                        <span class="badge <?php echo $education_count > 0 ? 'bg-success' : 'bg-secondary'; ?>">
                                            <?php echo $education_count; ?> <?php echo $education_count == 1 ? 'entry' : 'entries'; ?>
                                        </span>
                                    </div>
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <span><i class="fas fa-briefcase me-2 text-warning"></i>Experience</span>
                                        <span class="badge <?php echo $experience_count > 0 ? 'bg-success' : 'bg-secondary'; ?>">
                                            <?php echo $experience_count; ?> <?php echo $experience_count == 1 ? 'entry' : 'entries'; ?>
                                        </span>
                                    </div>
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <span><i class="fas fa-certificate me-2 text-danger"></i>Certificates</span>
                                        <span class="badge <?php echo $certificates_count > 0 ? 'bg-success' : 'bg-secondary'; ?>">
                                            <?php echo $certificates_count; ?> <?php echo $certificates_count == 1 ? 'cert' : 'certs'; ?>
                                        </span>
                                    </div>
                                    <div class="d-flex justify-content-between align-items-center mb-3">
                                        <span><i class="fas fa-users me-2 text-success"></i>References</span>
                                        <span class="badge <?php echo $references_count > 0 ? 'bg-success' : 'bg-secondary'; ?>">
                                            <?php echo $references_count; ?> <?php echo $references_count == 1 ? 'ref' : 'refs'; ?>
                                        </span>
                                    </div>
                                    <hr>
                                    <div class="d-flex justify-content-between align-items-center">
                                        <span><i class="fas fa-paper-plane me-2 text-info"></i>Applications</span>
                                        <span class="badge bg-info"><?php echo $application_stats['application_count']; ?></span>
                                    </div>
                                </div>
                            </div>

                            <!-- Quick Actions -->
                            <div class="card">
                                <div class="card-header">
                                    <h5 class="card-title mb-0">
                                        <i class="fas fa-bolt me-2"></i>Quick Actions
                                    </h5>
                                </div>
                                <div class="card-body">
                                    <div class="d-grid gap-2">
                                        <a href="cv-builder.php" class="btn btn-outline-primary">
                                            <i class="fas fa-file-alt me-2"></i>Build/Edit CV
                                        </a>
                                        <a href="../jobs.php" class="btn btn-outline-success">
                                            <i class="fas fa-search me-2"></i>Browse Jobs
                                        </a>
                                        <a href="applications.php" class="btn btn-outline-info">
                                            <i class="fas fa-paper-plane me-2"></i>View Applications
                                        </a>
                                        <a href="dashboard.php" class="btn btn-outline-secondary">
                                            <i class="fas fa-tachometer-alt me-2"></i>Back to Dashboard
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Nationality toggle functionality for profile edit
        function toggleNationalityFieldsEdit() {
            const southAfricanRadio = document.getElementById('south_african_edit');
            const foreignRadio = document.getElementById('foreign_edit');
            const southAfricanDiv = document.getElementById('south_african_id_edit');
            const foreignDiv = document.getElementById('foreign_passport_edit');
            const idInput = document.getElementById('id_number_edit');
            const passportInput = document.getElementById('passport_number_edit');

            if (southAfricanRadio.checked) {
                southAfricanDiv.style.display = 'block';
                foreignDiv.style.display = 'none';
                idInput.required = true;
                passportInput.required = false;
                passportInput.value = '';
            } else if (foreignRadio.checked) {
                southAfricanDiv.style.display = 'none';
                foreignDiv.style.display = 'block';
                idInput.required = false;
                passportInput.required = true;
                idInput.value = '';
            }
        }

        // Initialize nationality fields on page load
        document.addEventListener('DOMContentLoaded', function() {
            toggleNationalityFieldsEdit();
            
            // Add event listeners to nationality radio buttons
            document.getElementById('south_african_edit').addEventListener('change', toggleNationalityFieldsEdit);
            document.getElementById('foreign_edit').addEventListener('change', toggleNationalityFieldsEdit);
        });
    </script>
</body>
</html>