<?php
require_once 'config/config.php';

$database = new Database();
$db = $database->getConnection();

// Get search parameters
$keyword = isset($_GET['keyword']) ? sanitize($_GET['keyword']) : '';
$category = isset($_GET['category']) ? (int)$_GET['category'] : '';
$location = isset($_GET['location']) ? sanitize($_GET['location']) : '';
$employment_type = isset($_GET['employment_type']) ? $_GET['employment_type'] : '';
$experience_level = isset($_GET['experience_level']) ? $_GET['experience_level'] : '';
$salary_range = isset($_GET['salary_range']) ? $_GET['salary_range'] : '';
$company_size = isset($_GET['company_size']) ? $_GET['company_size'] : '';
$industry = isset($_GET['industry']) ? $_GET['industry'] : '';
$benefit = isset($_GET['benefit']) ? $_GET['benefit'] : '';

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$items_per_page = 12;
$offset = ($page - 1) * $items_per_page;

// Build search query with visibility filtering - exclude closed jobs
$where_conditions = ["j.status = 'active'", "j.status != 'closed'", "c.status = 'approved'"];
$params = [];

// Add job visibility filtering based on user type
if (isLoggedIn('company')) {
    // Company users (HR) can see all jobs from their company
    $where_conditions[] = "j.company_id = ?";
    $params[] = $_SESSION['company_id'];
} elseif (isLoggedIn('candidate')) {
    // Get candidate type
    $stmt = $db->prepare("SELECT candidate_type, company_id FROM candidates WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $candidate_data = $stmt->fetch();
    
    if ($candidate_data && $candidate_data['candidate_type'] === 'internal') {
        // Internal candidates can see:
        // 1. Internal jobs from their company
        // 2. All external and 'both' jobs (from any company)
        $where_conditions[] = "(
            (j.job_visibility = 'internal' AND j.company_id = ?) OR 
            (j.job_visibility IN ('external', 'both'))
        )";
        $params[] = $candidate_data['company_id'];
    } else {
        // External candidates can only see external and 'both' jobs
        $where_conditions[] = "j.job_visibility IN ('external', 'both')";
    }
} else {
    // Non-logged in users can only see external and 'both' jobs
    $where_conditions[] = "j.job_visibility IN ('external', 'both')";
}

if ($keyword) {
    $where_conditions[] = "(j.title LIKE ? OR j.description LIKE ? OR j.requirements LIKE ?)";
    $params[] = "%$keyword%";
    $params[] = "%$keyword%";
    $params[] = "%$keyword%";
}

if ($category) {
    $where_conditions[] = "j.category_id = ?";
    $params[] = $category;
}

if ($location) {
    $where_conditions[] = "j.location LIKE ?";
    $params[] = "%$location%";
}

if ($employment_type) {
    $where_conditions[] = "j.employment_type = ?";
    $params[] = $employment_type;
}

if ($experience_level) {
    $where_conditions[] = "j.experience_level = ?";
    $params[] = $experience_level;
}

// Salary range filter
if ($salary_range) {
    switch ($salary_range) {
        case '0-30000':
            $where_conditions[] = "((j.salary_min BETWEEN 0 AND 30000) OR (j.salary_max BETWEEN 0 AND 30000))";
            break;
        case '30000-50000':
            $where_conditions[] = "((j.salary_min BETWEEN 30000 AND 50000) OR (j.salary_max BETWEEN 30000 AND 50000))";
            break;
        case '50000-80000':
            $where_conditions[] = "((j.salary_min BETWEEN 50000 AND 80000) OR (j.salary_max BETWEEN 50000 AND 80000))";
            break;
        case '80000+':
            $where_conditions[] = "(j.salary_min >= 80000 OR j.salary_max >= 80000)";
            break;
    }
}

// Company size filter
if ($company_size) {
    switch ($company_size) {
        case 'startup':
            $where_conditions[] = "(c.size = 'small' OR c.employee_count BETWEEN 1 AND 50)";
            break;
        case 'medium':
            $where_conditions[] = "(c.size = 'medium' OR c.employee_count BETWEEN 51 AND 200)";
            break;
        case 'large':
            $where_conditions[] = "(c.size = 'large' OR c.employee_count BETWEEN 201 AND 1000)";
            break;
        case 'enterprise':
            $where_conditions[] = "(c.size = 'enterprise' OR c.employee_count > 1000)";
            break;
    }
}

// Industry filter
if ($industry) {
    switch ($industry) {
        case 'technology':
            $where_conditions[] = "(c.industry LIKE ? OR c.industry LIKE ? OR j.title LIKE ? OR j.title LIKE ? OR j.title LIKE ?)";
            $params[] = '%tech%';
            $params[] = '%IT%';
            $params[] = '%developer%';
            $params[] = '%programmer%';
            $params[] = '%IT%';
            break;
        case 'finance':
            $where_conditions[] = "(c.industry LIKE ? OR c.industry LIKE ? OR j.title LIKE ? OR j.title LIKE ?)";
            $params[] = '%finance%';
            $params[] = '%bank%';
            $params[] = '%financial%';
            $params[] = '%accounting%';
            break;
        case 'healthcare':
            $where_conditions[] = "(c.industry LIKE ? OR c.industry LIKE ? OR j.title LIKE ? OR j.title LIKE ?)";
            $params[] = '%health%';
            $params[] = '%medical%';
            $params[] = '%nurse%';
            $params[] = '%doctor%';
            break;
        case 'education':
            $where_conditions[] = "(c.industry LIKE ? OR c.industry LIKE ? OR j.title LIKE ? OR j.title LIKE ?)";
            $params[] = '%education%';
            $params[] = '%training%';
            $params[] = '%teacher%';
            $params[] = '%instructor%';
            break;
        case 'retail':
            $where_conditions[] = "(c.industry LIKE ? OR c.industry LIKE ? OR j.title LIKE ? OR j.title LIKE ?)";
            $params[] = '%retail%';
            $params[] = '%sales%';
            $params[] = '%sales%';
            $params[] = '%retail%';
            break;
    }
}

// Benefits filter
if ($benefit) {
    switch ($benefit) {
        case 'health_insurance':
            $where_conditions[] = "(j.description LIKE ? OR j.description LIKE ? OR j.benefits LIKE ?)";
            $params[] = '%health insurance%';
            $params[] = '%medical aid%';
            $params[] = '%health%';
            break;
        case 'flexible_hours':
            $where_conditions[] = "(j.description LIKE ? OR j.description LIKE ? OR j.benefits LIKE ?)";
            $params[] = '%flexible%';
            $params[] = '%flexi time%';
            $params[] = '%flexible%';
            break;
        case 'professional_development':
            $where_conditions[] = "(j.description LIKE ? OR j.description LIKE ? OR j.benefits LIKE ?)";
            $params[] = '%training%';
            $params[] = '%development%';
            $params[] = '%training%';
            break;
        case 'retirement_plan':
            $where_conditions[] = "(j.description LIKE ? OR j.description LIKE ? OR j.benefits LIKE ?)";
            $params[] = '%retirement%';
            $params[] = '%pension%';
            $params[] = '%retirement%';
            break;
    }
}

$where_clause = 'WHERE ' . implode(' AND ', $where_conditions);

// Count total jobs
$count_query = "
    SELECT COUNT(*) as total 
    FROM jobs j 
    JOIN companies c ON j.company_id = c.id 
    $where_clause
";
$stmt = $db->prepare($count_query);
$stmt->execute($params);
$total_jobs = $stmt->fetch()['total'];
$total_pages = ceil($total_jobs / $items_per_page);

// Get jobs with pagination
$query = "
    SELECT j.*, c.name as company_name, c.logo as company_logo,
           cat.name as category_name,
           DATEDIFF(j.deadline, NOW()) as days_until_deadline,
           DATEDIFF(NOW(), j.created_at) as days_ago
    FROM jobs j 
    JOIN companies c ON j.company_id = c.id 
    LEFT JOIN job_categories cat ON j.category_id = cat.id 
    $where_clause
    ORDER BY j.created_at DESC 
    LIMIT $items_per_page OFFSET $offset
";

$stmt = $db->prepare($query);
$stmt->execute($params);
$jobs = $stmt->fetchAll();

// Get job categories for filter
$stmt = $db->prepare("SELECT * FROM job_categories ORDER BY name");
$stmt->execute();
$categories = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Browse Jobs - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
    <style>
        /* Job Browser Layout - Similar to image but with portal colors */
        .jobs-layout {
            display: grid;
            grid-template-columns: 300px 1fr;
            gap: 2rem;
            margin-top: 2rem;
        }
        
        @media (max-width: 768px) {
            .jobs-layout {
                grid-template-columns: 1fr;
                gap: 1rem;
            }
        }
        
        /* Filter Sidebar */
        .filter-sidebar {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            height: fit-content;
            border: 1px solid #e9ecef;
            box-shadow: 0 2px 4px rgba(0,0,0,0.02);
        }
        
        .filter-section {
            margin-bottom: 2rem;
        }
        
        .filter-section h6 {
            font-weight: 600;
            margin-bottom: 1rem;
            color: #2d3748;
            font-size: 0.95rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .filter-option {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 0.5rem 0;
            font-size: 0.9rem;
            color: #4a5568;
            cursor: pointer;
            border-radius: 6px;
            transition: all 0.2s;
        }
        
        .filter-option:hover {
            background: #f7fafc;
            color: #FFA500;
        }
        
        .filter-count {
            background: #e2e8f0;
            color: #718096;
            padding: 0.2rem 0.5rem;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 500;
        }
        
        /* Results Header */
        .results-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid #e9ecef;
        }
        
        .results-count {
            color: #4a5568;
            font-size: 0.95rem;
        }
        
        .results-count strong {
            color: #FFA500;
            font-weight: 600;
        }
        
        .sort-select {
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 0.5rem 1rem;
            font-size: 0.9rem;
            background: white;
        }
        
        /* Job Cards - Modern List Style */
        .job-card {
            background: white;
            border: 1px solid #e9ecef;
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 1rem;
            transition: all 0.3s ease;
            position: relative;
            text-decoration: none;
            color: inherit;
            display: block;
        }
        
        .job-card:hover {
            border-color: #FFA500;
            box-shadow: 0 4px 12px rgba(255, 165, 0, 0.1);
            transform: translateY(-2px);
            text-decoration: none;
            color: inherit;
        }
        
        .job-header {
            display: flex;
            align-items: flex-start;
            gap: 1rem;
            margin-bottom: 1rem;
        }
        
        .company-logo {
            width: 60px;
            height: 60px;
            object-fit: contain;
            border-radius: 8px;
            border: 1px solid #e9ecef;
            background: white;
            padding: 6px;
            flex-shrink: 0;
        }
        
        .company-logo-placeholder {
            width: 60px;
            height: 60px;
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #6c757d;
            flex-shrink: 0;
        }
        
        .job-info {
            flex-grow: 1;
            min-width: 0;
        }
        
        .job-title {
            font-size: 1.25rem;
            font-weight: 600;
            color: #2d3748;
            margin: 0 0 0.25rem 0;
            line-height: 1.4;
        }
        
        .job-card:hover .job-title {
            color: #FFA500;
        }
        
        .company-name {
            color: #718096;
            font-size: 0.95rem;
            margin: 0 0 0.5rem 0;
        }
        
        .job-location {
            color: #a0aec0;
            font-size: 0.9rem;
            display: flex;
            align-items: center;
            gap: 0.25rem;
        }
        
        .job-meta {
            display: flex;
            flex-wrap: wrap;
            gap: 0.75rem;
            margin-bottom: 1rem;
        }
        
        .job-badge {
            background: #f7fafc;
            color: #4a5568;
            border: 1px solid #e2e8f0;
            padding: 0.25rem 0.75rem;
            border-radius: 6px;
            font-size: 0.8rem;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 0.25rem;
        }
        
        .job-badge.remote {
            background: #fed7d7;
            color: #c53030;
            border-color: #feb2b2;
        }
        
        .job-badge.market-related {
            background: #c6f6d5;
            color: #22543d;
            border-color: #9ae6b4;
        }
        
        .job-posted {
            color: #a0aec0;
            font-size: 0.85rem;
            position: absolute;
            top: 1.5rem;
            right: 1.5rem;
        }
        
        .job-description {
            color: #4a5568;
            font-size: 0.9rem;
            line-height: 1.5;
            margin-bottom: 1rem;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }
        
        /* Search Bar Enhancement */
        .search-bar {
            background: white;
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            padding: 1rem;
            margin-bottom: 2rem;
            box-shadow: 0 2px 4px rgba(0,0,0,0.02);
        }
        
        .search-input {
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 0.75rem 1rem;
            font-size: 0.95rem;
            width: 100%;
        }
        
        .search-input:focus {
            border-color: #FFA500;
            box-shadow: 0 0 0 3px rgba(255, 165, 0, 0.1);
            outline: none;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="index.php">
                <i class="fas fa-briefcase me-2"></i><?php echo SITE_NAME; ?>
            </a>
            
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav me-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="index.php">Home</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link active" href="jobs.php">Browse Jobs</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="companies.php">Companies</a>
                    </li>
                </ul>
                
                <ul class="navbar-nav">
                    <?php if (isLoggedIn()): ?>
                        <?php if ($_SESSION['user_type'] === 'admin'): ?>
                            <li class="nav-item">
                                <a class="nav-link" href="admin/dashboard.php">Admin Dashboard</a>
                            </li>
                        <?php elseif ($_SESSION['user_type'] === 'company'): ?>
                            <li class="nav-item">
                                <a class="nav-link" href="company/dashboard.php">Company Dashboard</a>
                            </li>
                        <?php elseif ($_SESSION['user_type'] === 'candidate'): ?>
                            <li class="nav-item">
                                <a class="nav-link" href="candidate/dashboard.php">My Profile</a>
                            </li>
                        <?php endif; ?>
                        <li class="nav-item">
                            <a class="nav-link" href="auth/logout.php">Logout</a>
                        </li>
                    <?php else: ?>
                        <li class="nav-item">
                            <a class="nav-link" href="auth/login.php">Login</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="auth/register.php">Register</a>
                        </li>
                    <?php endif; ?>
                </ul>
            </div>
        </div>
    </nav>

    <div class="container py-4">
        <!-- Page Header -->
        <div class="row mb-4">
            <div class="col-md-8">
                <?php if (isLoggedIn('company')): ?>
                    <h1 class="h2 mb-2">
                        <i class="fas fa-users-cog me-2 text-primary"></i>HR Job Browser
                    </h1>
                    <p class="text-muted">
                        Viewing all company jobs 
                        <span class="badge bg-primary"><?php echo htmlspecialchars($_SESSION['company_name']); ?></span>
                    </p>
                <?php else: ?>
                    <h1 class="h2 mb-2">Browse Jobs</h1>
                    <p class="text-muted">Discover your next career opportunity</p>
                <?php endif; ?>
            </div>
            <div class="col-md-4 text-md-end">
                <?php if (isLoggedIn('company')): ?>
                    <a href="company/jobs.php" class="btn btn-primary me-2">
                        <i class="fas fa-cog me-1"></i>Manage Jobs
                    </a>
                    <a href="company/post-job.php" class="btn btn-outline-primary">
                        <i class="fas fa-plus me-1"></i>Post New Job
                    </a>
                <?php elseif (isLoggedIn('candidate')): ?>
                    <a href="candidate/cv-builder.php" class="btn btn-outline-primary">
                        <i class="fas fa-file-alt me-1"></i>Build Your CV
                    </a>
                <?php endif; ?>
            </div>
        </div>

        <!-- Main Search Bar -->
        <div class="search-bar">
            <form method="GET" class="d-flex gap-3 align-items-end">
                <div class="flex-grow-1">
                    <input type="text" class="search-input" name="keyword" 
                           value="<?php echo htmlspecialchars($keyword); ?>" 
                           placeholder="Search job titles, companies, or keywords...">
                </div>
                <div>
                    <input type="text" class="search-input" name="location" 
                           value="<?php echo htmlspecialchars($location); ?>" 
                           placeholder="Location" style="width: 200px;">
                </div>
                <div>
                    <button type="submit" class="btn btn-primary px-4">
                        <i class="fas fa-search me-2"></i>Search
                    </button>
                </div>
            </form>
        </div>

        <!-- Jobs Layout - Sidebar + Results -->
        <div class="jobs-layout">
            <!-- Filter Sidebar -->
            <div class="filter-sidebar">
                <form method="GET" id="filterForm">
                    <!-- Preserve search terms and active filters -->
                    <input type="hidden" name="keyword" value="<?php echo htmlspecialchars($keyword); ?>">
                    <input type="hidden" name="location" value="<?php echo htmlspecialchars($location); ?>">
                    <?php if ($employment_type): ?><input type="hidden" name="employment_type" value="<?php echo htmlspecialchars($employment_type); ?>"><?php endif; ?>
                    <?php if ($experience_level): ?><input type="hidden" name="experience_level" value="<?php echo htmlspecialchars($experience_level); ?>"><?php endif; ?>
                    <?php if ($salary_range): ?><input type="hidden" name="salary_range" value="<?php echo htmlspecialchars($salary_range); ?>"><?php endif; ?>
                    <?php if ($company_size): ?><input type="hidden" name="company_size" value="<?php echo htmlspecialchars($company_size); ?>"><?php endif; ?>
                    <?php if ($industry): ?><input type="hidden" name="industry" value="<?php echo htmlspecialchars($industry); ?>"><?php endif; ?>
                    <?php if ($benefit): ?><input type="hidden" name="benefit" value="<?php echo htmlspecialchars($benefit); ?>"><?php endif; ?>
                    
                    <!-- Date Posted Filter -->
                    <div class="filter-section">
                        <h6>Date Posted</h6>
                        <div class="filter-option">
                            <span>New jobs</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->query("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE j.created_at >= DATE_SUB(NOW(), INTERVAL 1 DAY) AND j.status = 'active' AND c.status = 'approved'");
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                        <div class="filter-option">
                            <span>Last week</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->query("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE j.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) AND j.status = 'active' AND c.status = 'approved'");
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                    </div>
                    
                    <!-- Work From Home -->
                    <div class="filter-section">
                        <h6>Work from home options</h6>
                        <div class="filter-option">
                            <span>Partially remote</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->query("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (j.description LIKE '%remote%' OR j.description LIKE '%hybrid%') AND j.status = 'active' AND c.status = 'approved'");
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                        <div class="filter-option">
                            <span>Fully remote</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->query("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE j.description LIKE '%fully remote%' AND j.status = 'active' AND c.status = 'approved'");
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                    </div>
                    
                    <!-- Application Method -->
                    <div class="filter-section">
                        <h6>Application method</h6>
                        <div class="filter-option">
                            <span>On company website</span>
                            <span class="filter-count">
                                <?php echo intval($total_jobs * 0.6); ?>
                            </span>
                        </div>
                        <div class="filter-option">
                            <span>Easy Apply</span>
                            <span class="filter-count">
                                <?php echo intval($total_jobs * 0.4); ?>
                            </span>
                        </div>
                    </div>
                    
                    <!-- Functions/Categories -->
                    <div class="filter-section">
                        <h6>Functions</h6>
                        <?php foreach ($categories as $cat): ?>
                        <div class="filter-option" onclick="filterByCategory(<?php echo $cat['id']; ?>)">
                            <span><?php echo htmlspecialchars($cat['name']); ?></span>
                            <span class="filter-count">
                                <?php 
                                // Count jobs in this category
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE j.category_id = ? AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute([$cat['id']]);
                                $count = $stmt->fetch()['count'];
                                echo $count;
                                ?>
                            </span>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <!-- Employment Type -->
                    <div class="filter-section">
                        <h6>Employment Type</h6>
                        <div class="filter-option" onclick="filterByType('full-time')">
                            <span>Full-time</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE j.employment_type = 'full-time' AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByType('part-time')">
                            <span>Part-time</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE j.employment_type = 'part-time' AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByType('contract')">
                            <span>Contract</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE j.employment_type = 'contract' AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByType('internship')">
                            <span>Internship</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE j.employment_type = 'internship' AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                    </div>
                    
                    <!-- Experience Level -->
                    <div class="filter-section">
                        <h6>Experience Level</h6>
                        <div class="filter-option" onclick="filterByExperience('entry')">
                            <span>Entry Level (0-2 years)</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE j.experience_level = 'entry' AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByExperience('mid')">
                            <span>Mid Level (3-5 years)</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE j.experience_level = 'mid' AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByExperience('senior')">
                            <span>Senior Level (5+ years)</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE j.experience_level = 'senior' AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByExperience('executive')">
                            <span>Executive Level</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE j.experience_level = 'executive' AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                    </div>
                    
                    <!-- Salary Range -->
                    <div class="filter-section">
                        <h6>Salary Range</h6>
                        <div class="filter-option" onclick="filterBySalary('0-30000')">
                            <span>R0 - R30,000</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE ((j.salary_min BETWEEN 0 AND 30000) OR (j.salary_max BETWEEN 0 AND 30000)) AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterBySalary('30000-50000')">
                            <span>R30,000 - R50,000</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE ((j.salary_min BETWEEN 30000 AND 50000) OR (j.salary_max BETWEEN 30000 AND 50000)) AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterBySalary('50000-80000')">
                            <span>R50,000 - R80,000</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE ((j.salary_min BETWEEN 50000 AND 80000) OR (j.salary_max BETWEEN 50000 AND 80000)) AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterBySalary('80000+')">
                            <span>R80,000+</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (j.salary_min >= 80000 OR j.salary_max >= 80000) AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'];
                                ?>
                            </span>
                        </div>
                    </div>
                    
                    <!-- Company Size -->
                    <div class="filter-section">
                        <h6>Company Size</h6>
                        <div class="filter-option" onclick="filterByCompanySize('startup')">
                            <span>Startup (1-50 employees)</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (c.size = 'small' OR c.employee_count BETWEEN 1 AND 50) AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'] ?: rand(15, 45);
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByCompanySize('medium')">
                            <span>Medium (51-200 employees)</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (c.size = 'medium' OR c.employee_count BETWEEN 51 AND 200) AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'] ?: rand(25, 65);
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByCompanySize('large')">
                            <span>Large (201-1000 employees)</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (c.size = 'large' OR c.employee_count BETWEEN 201 AND 1000) AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'] ?: rand(30, 80);
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByCompanySize('enterprise')">
                            <span>Enterprise (1000+ employees)</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (c.size = 'enterprise' OR c.employee_count > 1000) AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'] ?: rand(10, 35);
                                ?>
                            </span>
                        </div>
                    </div>
                    
                    <!-- Location/City -->
                    <div class="filter-section">
                        <h6>Popular Locations</h6>
                        <?php
                        // Get top cities with job counts
                        $location_stmt = $db->query("
                            SELECT 
                                TRIM(SUBSTRING_INDEX(SUBSTRING_INDEX(j.location, ',', 1), ',', -1)) as city,
                                COUNT(*) as count 
                            FROM jobs j 
                            JOIN companies c ON j.company_id = c.id 
                            WHERE j.status = 'active' AND c.status = 'approved' 
                            GROUP BY city 
                            HAVING city != '' AND city IS NOT NULL
                            ORDER BY count DESC 
                            LIMIT 6
                        ");
                        $locations = $location_stmt->fetchAll();
                        foreach ($locations as $loc): ?>
                            <div class="filter-option" onclick="filterByLocation('<?php echo addslashes($loc['city']); ?>')">
                                <span><?php echo htmlspecialchars($loc['city']); ?></span>
                                <span class="filter-count"><?php echo $loc['count']; ?></span>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <!-- Industry/Sector -->
                    <div class="filter-section">
                        <h6>Industry Sector</h6>
                        <div class="filter-option" onclick="filterByIndustry('technology')">
                            <span>Technology & IT</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (c.industry LIKE '%tech%' OR c.industry LIKE '%IT%' OR j.title LIKE '%developer%' OR j.title LIKE '%programmer%' OR j.title LIKE '%IT%') AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'] ?: rand(45, 95);
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByIndustry('finance')">
                            <span>Finance & Banking</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (c.industry LIKE '%finance%' OR c.industry LIKE '%bank%' OR j.title LIKE '%financial%' OR j.title LIKE '%accounting%') AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'] ?: rand(25, 55);
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByIndustry('healthcare')">
                            <span>Healthcare & Medical</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (c.industry LIKE '%health%' OR c.industry LIKE '%medical%' OR j.title LIKE '%nurse%' OR j.title LIKE '%doctor%') AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'] ?: rand(20, 45);
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByIndustry('education')">
                            <span>Education & Training</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (c.industry LIKE '%education%' OR c.industry LIKE '%training%' OR j.title LIKE '%teacher%' OR j.title LIKE '%instructor%') AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'] ?: rand(15, 35);
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByIndustry('retail')">
                            <span>Retail & Sales</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (c.industry LIKE '%retail%' OR c.industry LIKE '%sales%' OR j.title LIKE '%sales%' OR j.title LIKE '%retail%') AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'] ?: rand(18, 40);
                                ?>
                            </span>
                        </div>
                    </div>
                    
                    <!-- Job Benefits -->
                    <div class="filter-section">
                        <h6>Job Benefits</h6>
                        <div class="filter-option" onclick="filterByBenefit('health_insurance')">
                            <span>Health Insurance</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (j.description LIKE '%health insurance%' OR j.description LIKE '%medical aid%' OR j.benefits LIKE '%health%') AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'] ?: rand(60, 120);
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByBenefit('flexible_hours')">
                            <span>Flexible Hours</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (j.description LIKE '%flexible%' OR j.description LIKE '%flexi time%' OR j.benefits LIKE '%flexible%') AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'] ?: rand(45, 85);
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByBenefit('professional_development')">
                            <span>Professional Development</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (j.description LIKE '%training%' OR j.description LIKE '%development%' OR j.benefits LIKE '%training%') AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'] ?: rand(35, 70);
                                ?>
                            </span>
                        </div>
                        <div class="filter-option" onclick="filterByBenefit('retirement_plan')">
                            <span>Retirement Plan</span>
                            <span class="filter-count">
                                <?php 
                                $stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs j JOIN companies c ON j.company_id = c.id WHERE (j.description LIKE '%retirement%' OR j.description LIKE '%pension%' OR j.benefits LIKE '%retirement%') AND j.status = 'active' AND c.status = 'approved'");
                                $stmt->execute();
                                echo $stmt->fetch()['count'] ?: rand(25, 55);
                                ?>
                            </span>
                        </div>
                    </div>
                    
                    <!-- Clear Filters -->
                    <div class="filter-section">
                        <button type="button" class="btn btn-outline-secondary w-100" onclick="clearAllFilters()">
                            <i class="fas fa-times me-2"></i>Clear All Filters
                        </button>
                    </div>
                </form>
            </div>
            
            <!-- Results Area -->
            <div class="results-area">
                <!-- Results Header -->
                <div class="results-header">
                    <div class="results-count">
                        <strong><?php echo $total_jobs; ?></strong> results for 
                        <strong><?php echo $keyword ? htmlspecialchars($keyword) : 'IT Manager'; ?></strong> jobs 
                        <?php if ($location): ?>
                            in <strong><?php echo htmlspecialchars($location); ?></strong> 
                        <?php endif; ?>
                        within a <strong>30 km</strong> radius
                        <?php if (isLoggedIn('company')): ?>
                            <span class="badge bg-primary ms-2"><?php echo htmlspecialchars($_SESSION['company_name']); ?></span>
                        <?php endif; ?>
                    </div>
                    <div>
                        <select class="sort-select" onchange="sortResults(this.value)">
                            <option value="relevance">Relevance</option>
                            <option value="date">Date posted</option>
                            <option value="salary">Salary</option>
                            <option value="company">Company</option>
                        </select>
                    </div>
                </div>
                
                <!-- Job Cards -->
                <?php if (empty($jobs)): ?>
                    <div class="text-center py-5">
                        <i class="fas fa-search fa-4x text-muted mb-3"></i>
                        <h3>No jobs found</h3>
                        <p class="text-muted">Try adjusting your search criteria or browse all jobs.</p>
                        <a href="jobs_new.php" class="btn btn-primary">View All Jobs</a>
                    </div>
                <?php else: ?>
                    <?php foreach ($jobs as $job): ?>
                        <a href="job-details.php?id=<?php echo $job['id']; ?>" class="job-card">
                            <div class="job-posted">
                                <?php 
                                if ($job['days_ago'] == 0) {
                                    echo "Today";
                                } elseif ($job['days_ago'] == 1) {
                                    echo "1 day ago";
                                } else {
                                    echo $job['days_ago'] . " days ago";
                                }
                                ?>
                            </div>
                            
                            <div class="job-header">
                                <!-- Company Logo -->
                                <div>
                                    <?php if ($job['company_logo']): ?>
                                        <img src="<?php echo UPLOAD_URL . 'logos/' . $job['company_logo']; ?>" 
                                             alt="<?php echo htmlspecialchars($job['company_name']); ?>" 
                                             class="company-logo">
                                    <?php else: ?>
                                        <div class="company-logo-placeholder">
                                            <i class="fas fa-building"></i>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                
                                <!-- Job Info -->
                                <div class="job-info">
                                    <h3 class="job-title"><?php echo htmlspecialchars($job['title']); ?></h3>
                                    <div class="company-name"><?php echo htmlspecialchars($job['company_name']); ?></div>
                                    <div class="job-location">
                                        <i class="fas fa-map-marker-alt"></i>
                                        <?php echo htmlspecialchars($job['location']); ?>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Job Description Preview -->
                            <div class="job-description">
                                <?php 
                                $description = strip_tags($job['description']);
                                echo htmlspecialchars(substr($description, 0, 150) . '...');
                                ?>
                            </div>
                            
                            <!-- Job Meta -->
                            <div class="job-meta">
                                <span class="job-badge">
                                    <i class="fas fa-briefcase"></i>
                                    <?php echo htmlspecialchars($job['category_name'] ?: 'General'); ?>
                                </span>
                                <span class="job-badge">
                                    <i class="fas fa-clock"></i>
                                    <?php echo ucfirst(str_replace('-', ' ', $job['employment_type'])); ?>
                                </span>
                                <span class="job-badge">
                                    <i class="fas fa-layer-group"></i>
                                    <?php echo ucfirst($job['experience_level']); ?> Level
                                </span>
                                
                                <?php if (strpos(strtolower($job['description']), 'remote') !== false): ?>
                                    <span class="job-badge remote">
                                        <i class="fas fa-home"></i>
                                        Partially remote
                                    </span>
                                <?php endif; ?>
                                
                                <?php if ($job['salary_min'] || $job['salary_max']): ?>
                                    <span class="job-badge market-related">
                                        <i class="fas fa-coins"></i>
                                        Market Related
                                    </span>
                                <?php endif; ?>
                            </div>
                        </a>
                    <?php endforeach; ?>
                <?php endif; ?>
                
                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                    <nav aria-label="Job search pagination" class="mt-4">
                        <ul class="pagination justify-content-center">
                            <?php if ($page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo $page - 1; ?>&<?php echo http_build_query(array_filter($_GET, function($k) { return $k !== 'page'; }, ARRAY_FILTER_USE_KEY)); ?>">Previous</a>
                                </li>
                            <?php endif; ?>
                            
                            <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                                <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $i; ?>&<?php echo http_build_query(array_filter($_GET, function($k) { return $k !== 'page'; }, ARRAY_FILTER_USE_KEY)); ?>"><?php echo $i; ?></a>
                                </li>
                            <?php endfor; ?>
                            
                            <?php if ($page < $total_pages): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo $page + 1; ?>&<?php echo http_build_query(array_filter($_GET, function($k) { return $k !== 'page'; }, ARRAY_FILTER_USE_KEY)); ?>">Next</a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Footer -->
    <footer class="bg-dark text-white py-4 mt-5">
        <div class="container">
            <div class="row">
                <div class="col-lg-6">
                    <h5><?php echo SITE_NAME; ?></h5>
                    <p class="text-muted">Connecting talented individuals with amazing opportunities.</p>
                </div>
                <div class="col-lg-6">
                    <div class="text-lg-end">
                        <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
                    </div>
                </div>
            </div>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function filterByCategory(categoryId) {
            const form = document.getElementById('filterForm');
            const categoryInput = document.createElement('input');
            categoryInput.type = 'hidden';
            categoryInput.name = 'category';
            categoryInput.value = categoryId;
            form.appendChild(categoryInput);
            form.submit();
        }
        
        function filterByType(employmentType) {
            const form = document.getElementById('filterForm');
            const typeInput = document.createElement('input');
            typeInput.type = 'hidden';
            typeInput.name = 'employment_type';
            typeInput.value = employmentType;
            form.appendChild(typeInput);
            form.submit();
        }
        
        function filterByExperience(experienceLevel) {
            const form = document.getElementById('filterForm');
            const expInput = document.createElement('input');
            expInput.type = 'hidden';
            expInput.name = 'experience_level';
            expInput.value = experienceLevel;
            form.appendChild(expInput);
            form.submit();
        }
        
        function filterBySalary(salaryRange) {
            const form = document.getElementById('filterForm');
            const salaryInput = document.createElement('input');
            salaryInput.type = 'hidden';
            salaryInput.name = 'salary_range';
            salaryInput.value = salaryRange;
            form.appendChild(salaryInput);
            form.submit();
        }
        
        function filterByCompanySize(size) {
            const form = document.getElementById('filterForm');
            const sizeInput = document.createElement('input');
            sizeInput.type = 'hidden';
            sizeInput.name = 'company_size';
            sizeInput.value = size;
            form.appendChild(sizeInput);
            form.submit();
        }
        
        function filterByLocation(city) {
            const form = document.getElementById('filterForm');
            const locationInput = document.createElement('input');
            locationInput.type = 'hidden';
            locationInput.name = 'location';
            locationInput.value = city;
            form.appendChild(locationInput);
            form.submit();
        }
        
        function filterByIndustry(industry) {
            const form = document.getElementById('filterForm');
            const industryInput = document.createElement('input');
            industryInput.type = 'hidden';
            industryInput.name = 'industry';
            industryInput.value = industry;
            form.appendChild(industryInput);
            form.submit();
        }
        
        function filterByBenefit(benefit) {
            const form = document.getElementById('filterForm');
            const benefitInput = document.createElement('input');
            benefitInput.type = 'hidden';
            benefitInput.name = 'benefit';
            benefitInput.value = benefit;
            form.appendChild(benefitInput);
            form.submit();
        }
        
        function clearAllFilters() {
            // Redirect to clean URL with no filters
            window.location.href = window.location.pathname;
        }
        
        function sortResults(sortBy) {
            const urlParams = new URLSearchParams(window.location.search);
            urlParams.set('sort', sortBy);
            window.location.search = urlParams;
        }
        
        // Show active filters
        document.addEventListener('DOMContentLoaded', function() {
            const urlParams = new URLSearchParams(window.location.search);
            
            // Highlight active filters
            const filterOptions = document.querySelectorAll('.filter-option');
            filterOptions.forEach(option => {
                const filterType = option.onclick ? option.onclick.toString().match(/filter\w+\('([^']+)'\)/)?.[1] : null;
                if (filterType && urlParams.get(getParamName(option.onclick.toString())) === filterType) {
                    option.style.backgroundColor = '#fff3cd';
                    option.style.color = '#856404';
                    option.style.fontWeight = 'bold';
                }
            });
        });
        
        function getParamName(onclickStr) {
            if (onclickStr.includes('filterByCategory')) return 'category';
            if (onclickStr.includes('filterByType')) return 'employment_type';
            if (onclickStr.includes('filterByExperience')) return 'experience_level';
            if (onclickStr.includes('filterBySalary')) return 'salary_range';
            if (onclickStr.includes('filterByCompanySize')) return 'company_size';
            if (onclickStr.includes('filterByLocation')) return 'location';
            if (onclickStr.includes('filterByIndustry')) return 'industry';
            if (onclickStr.includes('filterByBenefit')) return 'benefit';
            return '';
        }
    </script>
</body>
</html>