<?php
require_once 'config/config.php';

$database = new Database();
$db = $database->getConnection();

$company_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$company_id) {
    redirect('companies.php');
}

// Get company details
$stmt = $db->prepare("SELECT * FROM companies WHERE id = ? AND status = 'approved'");
$stmt->execute([$company_id]);
$company = $stmt->fetch();

if (!$company) {
    redirect('companies.php');
}

// Get company's active jobs
$stmt = $db->prepare("
    SELECT j.*, cat.name as category_name, COUNT(ja.id) as application_count
    FROM jobs j 
    LEFT JOIN job_categories cat ON j.category_id = cat.id 
    LEFT JOIN job_applications ja ON j.id = ja.job_id
    WHERE j.company_id = ? AND j.status = 'active'
    GROUP BY j.id
    ORDER BY j.created_at DESC
");
$stmt->execute([$company_id]);
$jobs = $stmt->fetchAll();

// Get company statistics
$stats = [];

// Total jobs posted
$stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs WHERE company_id = ?");
$stmt->execute([$company_id]);
$stats['total_jobs'] = $stmt->fetch()['count'];

// Active jobs
$stmt = $db->prepare("SELECT COUNT(*) as count FROM jobs WHERE company_id = ? AND status = 'active'");
$stmt->execute([$company_id]);
$stats['active_jobs'] = $stmt->fetch()['count'];

// Total applications received
$stmt = $db->prepare("
    SELECT COUNT(*) as count 
    FROM job_applications ja 
    JOIN jobs j ON ja.job_id = j.id 
    WHERE j.company_id = ?
");
$stmt->execute([$company_id]);
$stats['total_applications'] = $stmt->fetch()['count'];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($company['name']); ?> - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
    <style>
        .company-header {
            background: linear-gradient(135deg, #0d6efd, #0056b3);
            color: white;
            padding: 3rem 0;
            margin-bottom: 2rem;
        }
        .company-logo-large {
            width: 100px;
            height: 100px;
            object-fit: contain;
            border-radius: 10px;
            background: white;
            padding: 10px;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="index.php">
                <i class="fas fa-briefcase me-2"></i><?php echo SITE_NAME; ?>
            </a>
            
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav me-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="index.php">Home</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="jobs.php">Browse Jobs</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="companies.php">Companies</a>
                    </li>
                </ul>
                
                <ul class="navbar-nav">
                    <?php if (isLoggedIn()): ?>
                        <?php if ($_SESSION['user_type'] === 'admin'): ?>
                            <li class="nav-item">
                                <a class="nav-link" href="admin/dashboard.php">Admin Dashboard</a>
                            </li>
                        <?php elseif ($_SESSION['user_type'] === 'company'): ?>
                            <li class="nav-item">
                                <a class="nav-link" href="company/dashboard.php">Company Dashboard</a>
                            </li>
                        <?php elseif ($_SESSION['user_type'] === 'candidate'): ?>
                            <li class="nav-item">
                                <a class="nav-link" href="candidate/dashboard.php">My Profile</a>
                            </li>
                        <?php endif; ?>
                        <li class="nav-item">
                            <a class="nav-link" href="auth/logout.php">Logout</a>
                        </li>
                    <?php else: ?>
                        <li class="nav-item">
                            <a class="nav-link" href="auth/login.php">Login</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="auth/register.php">Register</a>
                        </li>
                    <?php endif; ?>
                </ul>
            </div>
        </div>
    </nav>

    <!-- Breadcrumb -->
    <div class="container">
        <nav aria-label="breadcrumb" class="pt-3">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="index.php">Home</a></li>
                <li class="breadcrumb-item"><a href="companies.php">Companies</a></li>
                <li class="breadcrumb-item active"><?php echo htmlspecialchars($company['name']); ?></li>
            </ol>
        </nav>
    </div>

    <!-- Company Header -->
    <div class="company-header">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <div class="d-flex align-items-center">
                        <div class="me-4">
                            <?php if ($company['logo']): ?>
                                <img src="<?php echo UPLOAD_URL . 'logos/' . $company['logo']; ?>" 
                                     alt="Company Logo" class="company-logo-large">
                            <?php else: ?>
                                <div class="company-logo-large bg-light d-flex align-items-center justify-content-center">
                                    <i class="fas fa-building fa-3x text-muted"></i>
                                </div>
                            <?php endif; ?>
                        </div>
                        <div>
                            <h1 class="h2 mb-2"><?php echo htmlspecialchars($company['name']); ?></h1>
                            <div class="d-flex flex-wrap gap-3 opacity-75">
                                <?php if ($company['website']): ?>
                                    <span>
                                        <i class="fas fa-globe me-2"></i>
                                        <a href="<?php echo htmlspecialchars($company['website']); ?>" target="_blank" class="text-white">
                                            <?php echo htmlspecialchars($company['website']); ?>
                                        </a>
                                    </span>
                                <?php endif; ?>
                                <?php if ($company['phone']): ?>
                                    <span>
                                        <i class="fas fa-phone me-2"></i><?php echo htmlspecialchars($company['phone']); ?>
                                    </span>
                                <?php endif; ?>
                                <span>
                                    <i class="fas fa-calendar me-2"></i>Member since <?php echo formatDate($company['created_at']); ?>
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 text-md-end">
                    <div class="d-flex justify-content-md-end gap-2">
                        <?php if ($stats['active_jobs'] > 0): ?>
                            <a href="jobs.php?company=<?php echo $company['id']; ?>" class="btn btn-light">
                                <i class="fas fa-briefcase me-2"></i>View Jobs (<?php echo $stats['active_jobs']; ?>)
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="container">
        <div class="row">
            <!-- Main Content -->
            <div class="col-lg-8 mb-4">
                <!-- Company Description -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-info-circle me-2"></i>About <?php echo htmlspecialchars($company['name']); ?>
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php if ($company['description']): ?>
                            <div class="content">
                                <?php echo nl2br(htmlspecialchars($company['description'])); ?>
                            </div>
                        <?php else: ?>
                            <p class="text-muted">No company description available.</p>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Active Jobs -->
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-briefcase me-2"></i>Current Job Openings
                        </h5>
                        <?php if ($stats['active_jobs'] > count($jobs)): ?>
                            <a href="jobs.php?company=<?php echo $company['id']; ?>" class="btn btn-sm btn-outline-primary">
                                View All Jobs
                            </a>
                        <?php endif; ?>
                    </div>
                    <div class="card-body">
                        <?php if (empty($jobs)): ?>
                            <div class="text-center py-4">
                                <i class="fas fa-briefcase fa-3x text-muted mb-3"></i>
                                <h5>No Active Job Openings</h5>
                                <p class="text-muted">This company currently has no active job postings.</p>
                                <a href="companies.php" class="btn btn-primary">Browse Other Companies</a>
                            </div>
                        <?php else: ?>
                            <div class="row">
                                <?php foreach ($jobs as $job): ?>
                                    <div class="col-md-6 mb-3">
                                        <div class="card job-card h-100">
                                            <div class="card-body">
                                                <h6 class="card-title">
                                                    <a href="job-details.php?id=<?php echo $job['id']; ?>" class="text-decoration-none">
                                                        <?php echo htmlspecialchars($job['title']); ?>
                                                    </a>
                                                </h6>
                                                
                                                <div class="mb-2">
                                                    <small class="text-muted">
                                                        <i class="fas fa-map-marker-alt me-1"></i><?php echo htmlspecialchars($job['location']); ?>
                                                    </small>
                                                </div>
                                                
                                                <div class="mb-2">
                                                    <span class="badge bg-primary"><?php echo ucfirst(str_replace('-', ' ', $job['employment_type'])); ?></span>
                                                    <?php if ($job['category_name']): ?>
                                                        <span class="badge bg-secondary ms-1"><?php echo htmlspecialchars($job['category_name']); ?></span>
                                                    <?php endif; ?>
                                                </div>
                                                
                                                <p class="card-text small"><?php echo substr(htmlspecialchars($job['description']), 0, 100) . '...'; ?></p>
                                                
                                                <?php if ($job['salary_min'] || $job['salary_max']): ?>
                                                    <div class="mb-2">
                                                        <small class="text-success fw-bold">
                                                            <i class="fas fa-dollar-sign"></i>
                                                            <?php if ($job['salary_min'] && $job['salary_max']): ?>
                                                                $<?php echo number_format($job['salary_min']); ?> - $<?php echo number_format($job['salary_max']); ?>
                                                            <?php elseif ($job['salary_min']): ?>
                                                                From $<?php echo number_format($job['salary_min']); ?>
                                                            <?php else: ?>
                                                                Up to $<?php echo number_format($job['salary_max']); ?>
                                                            <?php endif; ?>
                                                        </small>
                                                    </div>
                                                <?php endif; ?>
                                                
                                                <div class="d-flex justify-content-between align-items-center">
                                                    <small class="text-muted"><?php echo formatDate($job['created_at']); ?></small>
                                                    <small class="text-muted"><?php echo $job['application_count']; ?> applications</small>
                                                </div>
                                            </div>
                                            <div class="card-footer bg-transparent">
                                                <a href="job-details.php?id=<?php echo $job['id']; ?>" class="btn btn-outline-primary btn-sm w-100">
                                                    View Details
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Sidebar -->
            <div class="col-lg-4">
                <!-- Company Stats -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h6 class="card-title mb-0">
                            <i class="fas fa-chart-bar me-2"></i>Company Statistics
                        </h6>
                    </div>
                    <div class="card-body">
                        <div class="row text-center">
                            <div class="col-6 mb-3">
                                <div class="h4 text-primary mb-1"><?php echo $stats['total_jobs']; ?></div>
                                <small class="text-muted">Total Jobs Posted</small>
                            </div>
                            <div class="col-6 mb-3">
                                <div class="h4 text-success mb-1"><?php echo $stats['active_jobs']; ?></div>
                                <small class="text-muted">Active Openings</small>
                            </div>
                            <div class="col-12">
                                <div class="h4 text-info mb-1"><?php echo $stats['total_applications']; ?></div>
                                <small class="text-muted">Total Applications</small>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Company Contact -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h6 class="card-title mb-0">
                            <i class="fas fa-address-card me-2"></i>Contact Information
                        </h6>
                    </div>
                    <div class="card-body">
                        <?php if ($company['email']): ?>
                            <div class="mb-3">
                                <small class="text-muted">Email</small>
                                <div>
                                    <a href="mailto:<?php echo htmlspecialchars($company['email']); ?>">
                                        <?php echo htmlspecialchars($company['email']); ?>
                                    </a>
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($company['phone']): ?>
                            <div class="mb-3">
                                <small class="text-muted">Phone</small>
                                <div>
                                    <a href="tel:<?php echo htmlspecialchars($company['phone']); ?>">
                                        <?php echo htmlspecialchars($company['phone']); ?>
                                    </a>
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($company['address']): ?>
                            <div class="mb-3">
                                <small class="text-muted">Address</small>
                                <div><?php echo htmlspecialchars($company['address']); ?></div>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($company['website']): ?>
                            <div class="mb-0">
                                <small class="text-muted">Website</small>
                                <div>
                                    <a href="<?php echo htmlspecialchars($company['website']); ?>" target="_blank">
                                        <?php echo htmlspecialchars($company['website']); ?>
                                        <i class="fas fa-external-link-alt ms-1"></i>
                                    </a>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Back Button -->
                <div class="d-grid">
                    <a href="companies.php" class="btn btn-outline-secondary">
                        <i class="fas fa-arrow-left me-2"></i>Back to Companies
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Footer -->
    <footer class="bg-dark text-white py-4 mt-5">
        <div class="container">
            <div class="row">
                <div class="col-lg-6">
                    <h5><?php echo SITE_NAME; ?></h5>
                    <p class="text-muted">Connecting talented individuals with amazing opportunities.</p>
                </div>
                <div class="col-lg-6">
                    <div class="text-lg-end">
                        <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
                    </div>
                </div>
            </div>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>